<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package prime_education_learning
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function prime_education_learning_body_classes( $classes ) {
  global $prime_education_learning_post;
  
    if( !is_page_template( 'template-home.php' ) ){
        $classes[] = 'inner';
        // Adds a class of group-blog to blogs with more than 1 published author.
    }

    if ( is_multi_author() ) {
        $classes[] = 'group-blog ';
    }

    // Adds a class of custom-background-image to sites with a custom background image.
    if ( get_background_image() ) {
        $classes[] = 'custom-background-image';
    }
    
    // Adds a class of custom-background-color to sites with a custom background color.
    if ( get_background_color() != 'ffffff' ) {
        $classes[] = 'custom-background-color';
    }
    

    if( prime_education_learning_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() || 'product' === get_post_type() ) && ! is_active_sidebar( 'shop-sidebar' ) ){
        $classes[] = 'full-width';
    }    

    // Adds a class of hfeed to non-singular pages.
    if ( ! is_page() ) {
        $classes[] = 'hfeed ';
    }
  
    if( is_404() ||  is_search() ){
        $classes[] = 'full-width';
    }
  
    if( ! is_active_sidebar( 'right-sidebar' ) ) {
        $classes[] = 'full-width'; 
    }

    return $classes;
}
add_filter( 'body_class', 'prime_education_learning_body_classes' );

 /**
 * 
 * @link http://www.altafweb.com/2011/12/remove-specific-tag-from-php-string.html
 */
function prime_education_learning_strip_single( $tag, $string ){
    $string=preg_replace('/<'.$tag.'[^>]*>/i', '', $string);
    $string=preg_replace('/<\/'.$tag.'>/i', '', $string);
    return $string;
}

if ( ! function_exists( 'prime_education_learning_excerpt_more' ) ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... * 
 */
function prime_education_learning_excerpt_more($more) {
  return is_admin() ? $more : ' &hellip; ';
}
endif;
add_filter( 'excerpt_more', 'prime_education_learning_excerpt_more' );


if( ! function_exists( 'prime_education_learning_footer_credit' ) ):
/**
 * Footer Credits
*/
function prime_education_learning_footer_credit() {
    $prime_education_learning_copyright_text = get_theme_mod('prime_education_learning_footer_copyright_text');

    $prime_education_learning_text = '<div class="site-info"><div class="container"><span class="copyright">';
    if ($prime_education_learning_copyright_text) {
        $prime_education_learning_text .= wp_kses_post($prime_education_learning_copyright_text); 
    } else {
        $prime_education_learning_text .= esc_html__('&copy; ', 'prime-education-learning') . date_i18n(esc_html__('Y', 'prime-education-learning')); 
        $prime_education_learning_text .= ' <a href="' . esc_url(home_url('/')) . '">' . esc_html(get_bloginfo('name')) . '</a>' . esc_html__('. All Rights Reserved.', 'prime-education-learning');
    }
    $prime_education_learning_text .= '</span>';
    $prime_education_learning_text .= '<span class="by"> <a href="' . esc_url('https://www.themeignite.com/products/free-educational-wordpress-theme') . '" rel="nofollow" target="_blank">' . PRIME_EDUCATION_LEARNING_THEME_NAME . '</a>' . esc_html__(' By ', 'prime-education-learning') . '<a href="' . esc_url('https://themeignite.com/') . '" rel="nofollow" target="_blank">' . esc_html__('Themeignite', 'prime-education-learning') . '</a>.';
    $prime_education_learning_text .= sprintf(esc_html__(' Powered By %s', 'prime-education-learning'), '<a href="' . esc_url(__('https://wordpress.org/', 'prime-education-learning')) . '" target="_blank">WordPress</a>.');
    if (function_exists('the_privacy_policy_link')) {
        $prime_education_learning_text .= get_the_privacy_policy_link();
    }
    $prime_education_learning_text .= '</span></div></div>';
    echo apply_filters('prime_education_learning_footer_text', $prime_education_learning_text);
}
add_action('prime_education_learning_footer', 'prime_education_learning_footer_credit');
endif;

/**
 * Is Woocommerce activated
*/
if ( ! function_exists( 'prime_education_learning_woocommerce_activated' ) ) {
  function prime_education_learning_woocommerce_activated() {
    if ( class_exists( 'woocommerce' ) ) { return true; } else { return false; }
  }
}

if( ! function_exists( 'prime_education_learning_change_comment_form_default_fields' ) ) :
/**
 * Change Comment form default fields i.e. author, email & url.
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function prime_education_learning_change_comment_form_default_fields( $fields ){    
    // get the current commenter if available
    $prime_education_learning_commenter = wp_get_current_commenter();
 
    // core functionality
    $req      = get_option( 'require_name_email' );
    $prime_education_learning_aria_req = ( $req ? " aria-required='true'" : '' );
    $prime_education_learning_required = ( $req ? " required" : '' );
    $prime_education_learning_author   = ( $req ? __( 'Name*', 'prime-education-learning' ) : __( 'Name', 'prime-education-learning' ) );
    $prime_education_learning_email    = ( $req ? __( 'Email*', 'prime-education-learning' ) : __( 'Email', 'prime-education-learning' ) );
 
    // Change just the author field
    $fields['author'] = '<p class="comment-form-author"><label class="screen-reader-text" for="author">' . esc_html__( 'Name', 'prime-education-learning' ) . '<span class="required">*</span></label><input id="author" name="author" placeholder="' . esc_attr( $prime_education_learning_author ) . '" type="text" value="' . esc_attr( $prime_education_learning_commenter['comment_author'] ) . '" size="30"' . $prime_education_learning_aria_req . $prime_education_learning_required . ' /></p>';
    
    $fields['email'] = '<p class="comment-form-email"><label class="screen-reader-text" for="email">' . esc_html__( 'Email', 'prime-education-learning' ) . '<span class="required">*</span></label><input id="email" name="email" placeholder="' . esc_attr( $prime_education_learning_email ) . '" type="text" value="' . esc_attr(  $prime_education_learning_commenter['comment_author_email'] ) . '" size="30"' . $prime_education_learning_aria_req . $prime_education_learning_required. ' /></p>';
    
    $fields['url'] = '<p class="comment-form-url"><label class="screen-reader-text" for="url">' . esc_html__( 'Website', 'prime-education-learning' ) . '</label><input id="url" name="url" placeholder="' . esc_attr__( 'Website', 'prime-education-learning' ) . '" type="text" value="' . esc_attr( $prime_education_learning_commenter['comment_author_url'] ) . '" size="30" /></p>'; 
    
    return $fields;    
}
endif;
add_filter( 'comment_form_default_fields', 'prime_education_learning_change_comment_form_default_fields' );

if( ! function_exists( 'prime_education_learning_change_comment_form_defaults' ) ) :
/**
 * Change Comment Form defaults
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function prime_education_learning_change_comment_form_defaults( $defaults ){    
    $defaults['comment_field'] = '<p class="comment-form-comment"><label class="screen-reader-text" for="comment">' . esc_html__( 'Comment', 'prime-education-learning' ) . '</label><textarea id="comment" name="comment" placeholder="' . esc_attr__( 'Comment', 'prime-education-learning' ) . '" cols="45" rows="8" aria-required="true" required></textarea></p>';
    
    return $defaults;    
}
endif;
add_filter( 'comment_form_defaults', 'prime_education_learning_change_comment_form_defaults' );

if( ! function_exists( 'prime_education_learning_escape_text_tags' ) ) :
/**
 * Remove new line tags from string
 *
 * @param $text
 * @return string
 */
function prime_education_learning_escape_text_tags( $text ) {
    return (string) str_replace( array( "\r", "\n" ), '', strip_tags( $text ) );
}
endif;

if( ! function_exists( 'wp_body_open' ) ) :
/**
 * Fire the wp_body_open action.
 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
*/
function wp_body_open() {
    /**
     * Triggered after the opening <body> tag.
    */
    do_action( 'wp_body_open' );
}
endif;

if ( ! function_exists( 'prime_education_learning_get_fallback_svg' ) ) :    
/**
 * Get Fallback SVG
*/
function prime_education_learning_get_fallback_svg( $prime_education_learning_post_thumbnail ) {
    if( ! $prime_education_learning_post_thumbnail ){
        return;
    }
    
    $prime_education_learning_image_size = prime_education_learning_get_image_sizes( $prime_education_learning_post_thumbnail );
     
    if( $prime_education_learning_image_size ){ ?>
        <div class="svg-holder">
             <svg class="fallback-svg" viewBox="0 0 <?php echo esc_attr( $prime_education_learning_image_size['width'] ); ?> <?php echo esc_attr( $prime_education_learning_image_size['height'] ); ?>" preserveAspectRatio="none">
                    <rect width="<?php echo esc_attr( $prime_education_learning_image_size['width'] ); ?>" height="<?php echo esc_attr( $prime_education_learning_image_size['height'] ); ?>" style="fill:#dedddd;"></rect>
            </svg>
        </div>
        <?php
    }
}
endif;

function prime_education_learning_enqueue_google_fonts() {

    require get_template_directory() . '/inc/wptt-webfont-loader.php';

    wp_enqueue_style(
        'google-fonts-montserrat',
        wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Montserrat:ital,wght@0,100..900;1,100..900&display=swap' ),
        array(),
        '1.0'
    );

    wp_enqueue_style(
        'google-fonts-inter',
        wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Inter:ital,opsz,wght@0,14..32,100..900;1,14..32,100..900&display=swap' ),
        array(),
        '1.0'
    );
}
add_action( 'wp_enqueue_scripts', 'prime_education_learning_enqueue_google_fonts' );


if( ! function_exists( 'prime_education_learning_site_branding' ) ) :
/**
 * Site Branding
*/
function prime_education_learning_site_branding(){
    $prime_education_learning_logo_site_title = get_theme_mod( 'header_site_title', 1 );
    $prime_education_learning_tagline = get_theme_mod( 'header_tagline', false );
    $prime_education_learning_logo_width = get_theme_mod('logo_width', 100); // Retrieve the logo width setting

    ?>
    <div class="site-branding" style="max-width: <?php echo esc_attr(get_theme_mod('logo_width', '-1'))?>px;">
        <?php 
        // Check if custom logo is set and display it
        if (function_exists('has_custom_logo') && has_custom_logo()) {
            the_custom_logo();
        }
        if ($prime_education_learning_logo_site_title):
             if (is_front_page()): ?>
            <h1 class="site-title" style="font-size: <?php echo esc_attr(get_theme_mod('prime_education_learning_site_title_size', '30')); ?>px;">
            <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
          </h1>
            <?php else: ?>
                <p class="site-title" itemprop="name">
                    <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
                </p>
            <?php endif; ?>
        <?php endif; 
    
        if ($prime_education_learning_tagline) :
            $prime_education_learning_description = get_bloginfo('description', 'display');
            if ($prime_education_learning_description || is_customize_preview()) :
        ?>
                <p class="site-description" itemprop="description"><?php echo $prime_education_learning_description; ?></p>
            <?php endif;
        endif;
        ?>
    </div>
    <?php
}
endif;
if( ! function_exists( 'prime_education_learning_navigation' ) ) :
    /**
     * Site Navigation
    */
    function prime_education_learning_navigation(){
        ?>
        <nav class="main-navigation" id="site-navigation" role="navigation">
            <?php 
            wp_nav_menu( array( 
                'theme_location' => 'primary', 
                'menu_id' => 'primary-menu' 
            ) ); 
            ?>
        </nav>
        <?php
    }
endif;

if( ! function_exists( 'prime_education_learning_header' ) ) :
    /**
     * Header Start
    */
    function prime_education_learning_header(){
        $prime_education_learning_header_image = get_header_image();
        $prime_education_learning_sticky_header = get_theme_mod('prime_education_learning_sticky_header');
        $prime_education_learning_header_btn_text     = get_theme_mod( 'prime_education_learning_header_btn_text' );
        $prime_education_learning_header_btn_url     = get_theme_mod( 'prime_education_learning_header_btn_url' );
        $prime_education_learning_location     = get_theme_mod( 'prime_education_learning_header_location' );
        $prime_education_learning_phone        = get_theme_mod( 'prime_education_learning_header_phone' );
        $prime_education_learning_social_icon = get_theme_mod( 'prime_education_learning_social_icon_setting', false);
        $prime_education_learning_header_setting     = get_theme_mod( 'prime_education_learning_header_setting', false );
        $prime_education_learning_phone        = get_theme_mod( 'prime_education_learning_header_phone' );
        $prime_education_learning_email        = get_theme_mod( 'prime_education_learning_header_email' );
        $prime_education_learning_address     = get_theme_mod( 'prime_education_learning_address' );
        $prime_education_learning_timing     = get_theme_mod( 'prime_education_learning_header_timing' );
        ?>
        <div id="page-site-header" class="main-header">
            <header id="masthead" style="background-image: url('<?php echo esc_url( $prime_education_learning_header_image ); ?>');" class="site-header header-inner" role="banner">
                    <div class="topbar">
                        <div class="container">
                        <div class="row py-2 top-head">
                            <div class="col-xl-6 col-lg-6 col-md-6 align-self-center text-md-start">
                              <?php if ( $prime_education_learning_location ){?>
                                    <i class="fa-solid fa-stopwatch"></i><span>
                                        <?php echo esc_html( $prime_education_learning_location );?>
                                    </span>
                                <?php } ?> 
                            </div>
                            <div class="col-xl-6 col-lg-6 col-md-6 align-self-center header-detail text-md-end">
                               <?php if ( $prime_education_learning_social_icon ){?>
                                    <div class="social-links">
                                        <?php 
                                        $prime_education_learning_social_link1 = get_theme_mod( 'prime_education_learning_social_link_1' );
                                        $prime_education_learning_social_link2 = get_theme_mod( 'prime_education_learning_social_link_2' );
                                        $prime_education_learning_social_link3 = get_theme_mod( 'prime_education_learning_social_link_3' );
                                        $prime_education_learning_social_link4 = get_theme_mod( 'prime_education_learning_social_link_4' );

                                        if ( ! empty( $prime_education_learning_social_link1 ) ) {
                                            echo '<a class="social1" href="' . esc_url( $prime_education_learning_social_link1 ) . '" target="_blank"><i class="fab fa-facebook-f"></i></a>';
                                        }
                                        if ( ! empty( $prime_education_learning_social_link2 ) ) {
                                            echo '<a class="social2" href="' . esc_url( $prime_education_learning_social_link2 ) . '" target="_blank"><i class="fab fa-twitter"></i></a>';
                                        } 
                                        if ( ! empty( $prime_education_learning_social_link3 ) ) {
                                            echo '<a class="social3" href="' . esc_url( $prime_education_learning_social_link3 ) . '" target="_blank"><i class="fab fa-instagram"></i></a>';
                                        }
                                        if ( ! empty( $prime_education_learning_social_link4 ) ) {
                                            echo '<a class="social4" href="' . esc_url( $prime_education_learning_social_link4 ) . '" target="_blank"><i class="fab fa-pinterest-p"></i></a>';
                                        }
                                        ?>
                                    </div>
                                <?php } ?>  
                            </div>
                        </div>
                    </div>
                    </div>
                    <div class="mid-head">
                       <div class="container">
                        <div class="row">
                            <div class="col-xl-5 col-lg-3 col-md-3 align-self-center">
                                <?php prime_education_learning_site_branding(); ?>
                            </div>
                            <div class="col-xl-3 col-lg-4 col-md-3 align-self-center">
                                <?php if ( $prime_education_learning_address ){?>
                                    <div class="location">
                                        <span class="contact-icon">
                                            <i class="fas fa-map-marker-alt"></i>
                                        </span>
                                        <span class="contact-box">
                                            <p class="contact-heading"><?php echo esc_html('Address','prime-education-learning' );?></p>
                                            <span class="location-text"><?php echo esc_html( $prime_education_learning_address );?></span>
                                        </span>
                                    </div>
                                <?php } ?>     
                            </div>
                            <div class="col-xl-2 col-lg-3 col-md-3 align-self-center">
                                <?php if ( $prime_education_learning_email ){?>
                                    <div class="location">
                                        <span class="contact-icon">
                                            <i class="fas fa-phone-alt"></i>
                                        </span>
                                        <span class="contact-box">
                                            <p class="contact-heading"><?php echo esc_html('Mail Us','prime-education-learning' );?></p>
                                            <span class="location-text">
                                                <a href="mailto:<?php echo esc_attr($prime_education_learning_email);?>">
                                                    <?php echo esc_html($prime_education_learning_email);?>
                                                </a>
                                            </span>
                                        </span>
                                        
                                    </div>
                                <?php } ?>
                            </div>
                            <div class="col-xl-2 col-lg-2 col-md-3 align-self-center">
                                <?php if ( $prime_education_learning_phone ){?>
                                    <div class="location">
                                        <span class="contact-icon">
                                            <i class="far fa-clock"></i>
                                        </span>
                                        <span class="contact-box">
                                            <p class="contact-heading"><?php echo esc_html('Phone','prime-education-learning' );?></p>
                                             <span class="location-text">
                                                <a href="tel:<?php echo esc_attr($prime_education_learning_phone);?>">
                                                    <?php echo esc_html( $prime_education_learning_phone);?>
                                                </a>
                                            </span>
                                        </span>
                                    </div>
                                <?php } ?>
                            </div>
                        </div>
                    </div> 
                    </div>
                        <div class="theme-menu head_bg" data-sticky="<?php echo $prime_education_learning_sticky_header; ?>">
                            <div class="container">
                                <div class="row">
                                <div class="col-xl-9 col-lg-8 col-md-1 align-self-center">
                                   <?php prime_education_learning_navigation(); ?> 
                                </div>
                                <div class="col-xl-1 col-lg-1 col-md-6 align-self-center position-relative">
                                    <?php if( get_theme_mod('prime_education_learning_show_hide_search', false) ) { ?>
                                        <div class="search-body">
                                            <button type="button" class="search-show">
                                                <i class="<?php echo esc_attr(get_theme_mod('prime_education_learning_search_icon', 'fas fa-search')); ?>"></i>
                                            </button>
                                        </div>
                                        <div class="searchform-inner">
                                            <?php get_search_form(); ?>
                                            <button type="button" class="close" aria-label="<?php esc_attr_e( 'Close', 'prime-education-learning' ); ?>"><span aria-hidden="true">X</span></button>
                                        </div> 
                                    <?php } ?> 
                                </div>
                                <div class="col-xl-2 col-lg-3 col-md-5 align-self-center btn_bg">
                                    <?php if ( $prime_education_learning_header_btn_text ){?>
                                        <div class="menudiv-button">
                                            <a href="<?php echo esc_url($prime_education_learning_header_btn_url);?>"><i class="fa-solid fa-rocket"></i><?php echo esc_html($prime_education_learning_header_btn_text);?></a>
                                        </div>
                                    <?php } ?>
                                </div>
                            </div>
                            </div>
                        </div>
                    
            </header>
        </div>
        <?php
    }
endif;
add_action( 'prime_education_learning_header', 'prime_education_learning_header', 20 );
